package VPD;


# --------------------------------------------------------------------------------+
# -- Constructor --                                                               |
# --------------------------------------------------------------------------------+
sub new
{
  my $this = shift;
  my $class = ref($this) || $this;
  my $self = {};
  bless $self, $class;
  return $self;
}





# --------------------------------------------------------------------------------+
# -- Name     : addProperty                                                       |
# -- Function : Insert a unique Key with its corresponding value into this object |
# --             Duplicate key will be appended with a number --                  |
# --             Example: CB, CB~1, CB~2, CB~3, etc...                            |
# -- Input    : Key & Value pair                                                  |
# -- Output   : None                                                              |
# --------------------------------------------------------------------------------+
sub add_Property
{
  my ( $this ) = shift;
  my ( $inKey, $inValue ) = @_;

  if ( exists  $this->{$inKey} )  # -- Still need to store values of duplicate keys
  {
    # -- 1. Create / append a private Data counter ex: "~CB" for "CB" Keys --
    $this->{"~$inKey"} += 1;

    # -- 2. Store the Value into a Fake Key --
    my( $fakeKey )    = "$inKey" . "~" . $this->{"~$inKey"};   # -- Ex: "CB~2"
    $this->{$fakeKey} = $inValue;
  }
  else  # -- Key doesn't exists yet --
  {
    $this->{$inKey} = $inValue;
  }
}


# --------------------------------------------------------------------------------+
# -- Name     : add_lsvpdString                                                   |
# -- Function : Similar to addProperty but the input is lsvpd string              |
# -- Input    : ONE line of lsvpd string  (ex: *TM 7040-1234 )                    |
# -- Output   : None                                                              |
# --------------------------------------------------------------------------------+
sub add_lsvpdString
{
  my ( $this ) = shift;
  my ( $inString ) = @_;

  $value = $inString;
  $value =~ s/\*(..) //;   # -- Get rid of the Key --
  $key = $1;               # -- Assign Key from the Value --
  $value =~ s/\n$//;       # -- Take out trailling newline --

  $this->add_Property($key, $value);
}



# --------------------------------------------------------------------------------+
# -- Name     : get_Value                                                         |
# -- Function : Returns the value of a given Key in this VPD object               |
# -- Input    : Key                                                               |
# -- Output   : Value                                                             |
# --------------------------------------------------------------------------------+
sub get_Value
{
  my ( $this ) = shift;
  my ( $inKey ) = @_;

  return ( $this->{$inKey} )  if ( exists $this->{$inKey} );
}


# --------------------------------------------------------------------------------+
# -- Name     : has_Key                                                           |
# -- Function : Determines if this object contains given list of keys             |
# -- Input    : Key                                                               |
# -- Output   : 0 if Key is not Found                                             |
# --            1 if ALL specified Keys are Found                                 |
# --------------------------------------------------------------------------------+
sub has_Keys
{
  my ( $this ) = shift;
  my ( @inKeys ) = @_;

  foreach $key (@inKeys)
  {
    return (0) unless ( exists $this->{$key} );
  }

  return (1);  # -- ALL Keys have been found --
}




# --------------------------------------------------------------------------------+
# -- Name     : write_Object                                                      |
# -- Function : Writes the VPD object back into the lsvpd format                  |
# -- Input    : Output_Handle = Destination                                       |
# -- Output   :                                                                   |
# --              Note: To write to screen, use "\*STDOUT"                        |
# --                    For a file, open a file for writing, then pass the handle |
# --------------------------------------------------------------------------------+
sub write_Object
{
  my ( $this ) = shift;
  my ( $inOutputHandle ) = shift;  # -- Where the output will be written to --


  print $inOutputHandle "*FC ????????\n";   # -- Separator is written first --

  # ----------------------------------------
  KEY : foreach $key (sort keys %$this)
  {
    next KEY if ( $key =~ /^\~/ );  # -- ignore Fake Key counter --

    my($realKey) = $key;

    if ( length ($key) > 2 )   # -- lsvpd Keys have only 2 characters, 
    {                          # --  anything else must be Fake Keys... must be truncated --
      $realKey =~ s/~.+//;
    }
    print $inOutputHandle ("*" . $realKey . " " . "$this->{$key}" . "\n");
  }
}



# --------------------------------------------------------------------------------+
# -- Name     : _display                                                          |
# -- Function : Prints the VPD object (Debugging Purposes)                        |
# -- Input    : None (optional: list of keys to be printed)                       |
# -- Output   : Printout of the VPD object                                        |
# --------------------------------------------------------------------------------+
sub _display
{
  my $this = shift;
  my @inKeys = @_ ? @_ : sort keys %$this;

  my ($key) = ();

  foreach $key (@inKeys)
  {
    print "$key \t $this->{$key}\n";
  }
}



# --------------------------------------------------------------------------------+
# -- Name     : isVPD_Object                                                      |
# -- Function : Since Perl doesn't provide type checking like C++                 |
# --               only VPD objects can have this method                          |
# -- Input    : none                                                              |
# -- Output   : returns true                                                      |
# --------------------------------------------------------------------------------+
sub isVPD_Object
{
  return (1);
}


1;
